#!/usr/bin/env bash
set -u
set -o pipefail

log() { echo "[$(date +'%H:%M:%S')] $*"; }

wait_port() {
  local host="$1"
  local port="$2"
  local name="$3"
  local tries="${4:-60}"

  log "Waiting for $name on $host:$port ..."
  for i in $(seq 1 "$tries"); do
    if (echo >"/dev/tcp/${host}/${port}") >/dev/null 2>&1; then
      log "$name is UP on $host:$port"
      return 0
    fi
    sleep 1
  done
  log "ERROR: $name did not start (timeout) on $host:$port"
  return 1
}

hdfs_mkdir_if_missing() {
  local path="$1"
  hdfs dfs -test -d "$path" && return 0
  log "Creating HDFS dir: $path"
  hdfs dfs -mkdir -p "$path"
}

dump_logs() {
  log "==== tail /opt/hadoop/logs (last 200 lines each) ===="
  for f in /opt/hadoop/logs/*; do
    [ -f "$f" ] || continue
    echo "==> $f <=="
    tail -n 200 "$f" || true
    echo
  done
  log "====================================================="
}

log "== Hadoop one-container starting =="
log "Hostname: $(hostname)"
log "User: $(id -u):$(id -g) ($(whoami))"
log "JAVA_HOME: ${JAVA_HOME:-<unset>}"

mkdir -p /hadoop/dfs/name /hadoop/dfs/data
mkdir -p /opt/hadoop/logs

log "Fixing permissions on HDFS dirs..."
chmod -R 700 /hadoop/dfs/name /hadoop/dfs/data || true

# Formateo solo 1ª vez
if [ ! -f /hadoop/dfs/name/current/VERSION ]; then
  log "Formatting NameNode (first run) ..."
  hdfs namenode -format -force -nonInteractive || { log "FAILED formatting namenode"; dump_logs; exit 1; }
fi

log "Starting NameNode..."
hdfs --daemon start namenode || true
wait_port "hadoop-one" "9870" "NameNode UI (9870)" 90 || { dump_logs; exit 1; }

log "Starting DataNode..."
hdfs --daemon start datanode || true
wait_port "hadoop-one" "9864" "DataNode UI (9864)" 90 || { dump_logs; exit 1; }

log "Waiting for HDFS to respond..."
for i in $(seq 1 90); do
  if hdfs dfs -ls / >/dev/null 2>&1; then
    log "HDFS is responding."
    break
  fi
  sleep 1
done

log "Starting ResourceManager..."
yarn --daemon start resourcemanager || true
wait_port "hadoop-one" "8088" "ResourceManager UI (8088)" 90 || { dump_logs; exit 1; }

log "Starting NodeManager..."
yarn --daemon start nodemanager || true
wait_port "hadoop-one" "8042" "NodeManager UI (8042)" 90 || { dump_logs; exit 1; }

log "Preparing MapReduce JobHistory directories in HDFS..."
hdfs_mkdir_if_missing "/mr-history"
hdfs_mkdir_if_missing "/mr-history/tmp"
hdfs_mkdir_if_missing "/mr-history/done"
hdfs dfs -chmod -R 1777 /mr-history/tmp || true
hdfs dfs -chmod -R 755 /mr-history/done || true

# ---- Arrancar HistoryServer en foreground para ver el error real ----
log "Starting MapReduce HistoryServer (foreground)..."
# Esto escribe en /opt/hadoop/logs/hadoop-*-historyserver-*.log
# y si revienta veremos el motivo en logs.
nohup mapred historyserver >/opt/hadoop/logs/historyserver.stdout 2>/opt/hadoop/logs/historyserver.stderr &

# Esperar 19888 en varios hosts (en algunos casos no está en 127.0.0.1)
wait_port "hadoop-one" "19888" "JobHistory UI (19888)" 240 || {
  log "HistoryServer did not open 19888. Showing diagnostics..."
  log "--- netstat/ss ---"
  (ss -lntp 2>/dev/null || netstat -lntp 2>/dev/null || true)
  log "--- historyserver.stderr ---"
  tail -n 200 /opt/hadoop/logs/historyserver.stderr 2>/dev/null || true
  log "--- historyserver.stdout ---"
  tail -n 200 /opt/hadoop/logs/historyserver.stdout 2>/dev/null || true
  dump_logs
  exit 1
}

log "== Services started =="
log "UIs: NN http://localhost:9870 | RM http://localhost:8088 | NM http://localhost:8042 | HS http://localhost:19888"

exec tail -F /opt/hadoop/logs/*.log /opt/hadoop/logs/historyserver.* 2>/dev/null
